import 'dart:convert';
import 'dart:io';

import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:open_file/open_file.dart';
import 'package:path_provider/path_provider.dart';
import 'package:permission_handler/permission_handler.dart';
import 'package:viserpay/core/helper/string_format_helper.dart';
import 'package:viserpay/core/route/route.dart';
import 'package:viserpay/core/utils/url_container.dart';
import 'package:viserpay/core/utils/util.dart';
import 'package:viserpay/data/model/authorization/authorization_response_model.dart';
import 'package:viserpay/data/model/global/charge/charge_model.dart';
import 'package:viserpay/data/model/invoice/invoice_list_response_model.dart';
import 'package:viserpay/data/model/make_payment/make_payment_submit_response_modal.dart';
import 'package:http/http.dart' as http;

import '../../../core/utils/my_color.dart';
import '../../../core/utils/my_strings.dart';
import '../../../view/components/snack_bar/show_custom_snackbar.dart';
import '../../model/global/response_model/response_model.dart';
import '../../repo/invoice/invoice_repo.dart';

class InvoiceHistoryController extends GetxController {
  InvoiceRepo invoiceHistoryRepo;
  InvoiceHistoryController({required this.invoiceHistoryRepo});

  bool isLoading = true;
  InvoiceListResponseModel invoiceListResponseModel = InvoiceListResponseModel();
  TextEditingController pinController = TextEditingController();
  FocusNode pinFocusNode = FocusNode();
  int page = 0;
  String? nextPageUrl;

  List<InvoiceDataModel> invoiceList = [];

  Future<void> initialData() async {
    page = 0;
    invoiceList.clear();
    isLoading = true;
    update();

    await loadInvoiceData();
    isLoading = false;
    update();
  }

  Future<void> loadInvoiceData() async {
    page = page + 1;
    if (page == 1) {
      invoiceList.clear();
    }

    ResponseModel responseModel = await invoiceHistoryRepo.getAllInvoiceData(page);
    if (responseModel.statusCode == 200) {
      invoiceListResponseModel = invoiceListResponseModelFromJson(responseModel.responseJson);

      nextPageUrl = invoiceListResponseModel.data?.invoices?.nextPageUrl;
      if (invoiceListResponseModel.status.toString().toLowerCase() == MyStrings.success.toLowerCase()) {
        currentBalance = invoiceListResponseModel.data?.currentBalance.toString() ?? "0";

        otpTypeList.clear();
        otpTypeList.addAll(invoiceListResponseModel.data?.otpType?.toList() ?? []);
        makePaymentCharge = invoiceListResponseModel.data?.paymentCharge;
        percentCharge = invoiceListResponseModel.data?.paymentCharge?.userPaymentPercentCharge ?? "0.0";
        fixedCharge = invoiceListResponseModel.data?.paymentCharge?.userPaymentFixedCharge ?? "0.0";
        update();

        List<InvoiceDataModel>? tempInvoiceList = invoiceListResponseModel.data?.invoices?.data;
        if (tempInvoiceList != null && tempInvoiceList.isNotEmpty) {
          invoiceList.addAll(tempInvoiceList);
        }
      } else {
        CustomSnackBar.error(errorList: invoiceListResponseModel.message?.error ?? [MyStrings.somethingWentWrong]);
      }
    } else {
      CustomSnackBar.error(errorList: [responseModel.message]);
    }

    isLoading = false;
    update();
  }

  bool hasNext() {
    return nextPageUrl != null && nextPageUrl!.isNotEmpty && nextPageUrl != 'null' ? true : false;
  }

  dynamic getStatusOrColor(int index, {bool isStatus = true}) {
    String status = invoiceList[index].status ?? '';

    if (isStatus) {
      String text = status == "0"
          ? MyStrings.unpaid
          : status == "1"
              ? MyStrings.paid
              : status == "2"
                  ? MyStrings.rejected
                  : status == "3"
                      ? MyStrings.canceled
                      : "";
      return text;
    } else {
      Color color = status == "0"
          ? MyColor.highPriorityPurpleColor
          : status == "1"
              ? MyColor.colorGreen
              : status == "2"
                  ? MyColor.colorRed
                  : MyColor.colorOrange;
      return color;
    }
  }

  dynamic getPaymentStatusOrColor(int index, {bool isStatus = true}) {
    String paymentStatus = invoiceList[index].status ?? '';

    if (isStatus) {
      String text = paymentStatus == "0"
          ? MyStrings.unpaid
          : paymentStatus == "1"
              ? MyStrings.paid
              : "";
      return text;
    } else {
      Color color = paymentStatus == "0"
          ? MyColor.colorOrange
          : paymentStatus == "1"
              ? MyColor.colorGreen
              : MyColor.transparentColor;
      return color;
    }
  }

  List<String> otpTypeList = [];
  String selectedOtpType = "null";

  void selectOtpType(String otpType) {
    selectedOtpType = otpType;
    update();
  }

  String currentBalance = '0';
  GlobalCharge? makePaymentCharge;
  double mainAmount = 0;
  String charge = "";
  String totalCharge = "";
  String payableText = '';
  String currency = '';
  String currencySym = '';
  String percentCharge = "";
  String fixedCharge = "";

  void changeInfoWidget(String mainAmountText) {
    currency = invoiceHistoryRepo.apiClient.getCurrencyOrUsername(isSymbol: false);
    currencySym = invoiceHistoryRepo.apiClient.getCurrencyOrUsername(isSymbol: true);
    update();
    mainAmount = double.tryParse(mainAmountText) ?? 0.0;
    update();
    double percent = double.tryParse(percentCharge) ?? 0;

    double tempPercentCharge = mainAmount * percent / 100;
    double tempFixedCharge = double.tryParse(fixedCharge) ?? 0;
    double tempTotalCharge = tempPercentCharge + tempFixedCharge;
    double cap = double.tryParse(makePaymentCharge?.cap ?? "0") ?? 0;
    double mainCap = cap;
    if (cap != -1.0 && cap != 1 && tempTotalCharge > mainCap) {
      tempTotalCharge = mainCap;
    }
    charge = StringConverter.formatNumber('$tempTotalCharge', precision: 2);
    double payable = tempTotalCharge + mainAmount;
    payableText = StringConverter.formatNumber(payable.toString(), precision: 2);
    update();
  }

  bool validatePinCode() {
    if (pinController.text.length != 4) {
      MyUtils.vibrate();
      CustomSnackBar.error(errorList: [MyStrings.pinLengthErrorMessage]);
      return false;
    }
    if (pinController.text.isEmpty) {
      MyUtils.vibrate();
      CustomSnackBar.error(errorList: [MyStrings.pinErrorMessage]);
      return false;
    }

    return true;
  }

//submit
  bool isSubmitLoading = false;
  Future<void> submitMakePayment({required String invoiceID}) async {
    isSubmitLoading = true;
    update();
    try {
      ResponseModel responseModel = await invoiceHistoryRepo.submitInvoicePayment(
        otpType: selectedOtpType,
        pin: pinController.text,
        invoiceID: invoiceID,
      );
      if (responseModel.statusCode == 200) {
        MakePaymentSubmitResponseModal modal = MakePaymentSubmitResponseModal.fromJson(jsonDecode(responseModel.responseJson));

        if (modal.status == "success") {
          Get.back();

          if (modal.data?.actionID == 'null') {
            Get.offAllNamed(RouteHelper.makePaymentSuccessScreen, arguments: [responseModel]);
            CustomSnackBar.success(successList: [MyStrings.makePaymentSuccess]);
          } else {
            Get.offAllNamed(
              RouteHelper.otpScreen,
              arguments: [
                modal.data?.actionID,
                RouteHelper.makePaymentSuccessScreen,
                pinController.text.toString(),
                selectedOtpType,
              ],
            );
          }
        } else {
          Get.back();
          CustomSnackBar.error(errorList: modal.message?.error ?? [MyStrings.somethingWentWrong]);
        }
      } else {
        Get.back();
        CustomSnackBar.error(errorList: [responseModel.message]);
      }
    } catch (e) {
      isSubmitLoading = false;
      update();
    }
    isSubmitLoading = false;
    update();
  }

  bool isDiscardInvoiceLoading = false;

  Future<void> rejectInvoice(String rejectReason, String invoiceID) async {
    isDiscardInvoiceLoading = true;
    update();
    ResponseModel responseModel = await invoiceHistoryRepo.rejectInvoice(invoiceID, rejectReason);
    if (responseModel.statusCode == 200) {
      AuthorizationResponseModel model = AuthorizationResponseModel.fromJson(jsonDecode(responseModel.responseJson));
      if (model.status.toString().toLowerCase() == "success") {
        Get.back(result: 'success');
        CustomSnackBar.success(successList: model.message?.success ?? [MyStrings.requestSuccess]);
      } else {
        CustomSnackBar.error(errorList: model.message?.error ?? [MyStrings.requestFail]);
      }
    } else {
      CustomSnackBar.error(errorList: [responseModel.message]);
    }
    isDiscardInvoiceLoading = false;
    update();
  }

  //DOWNLOAD

  String downloadUrl = "";
  String downloadFileName = "";
  bool downloadLoading = false;
  Future<void> downloadINVOICE(String invoiceID) async {
    downloadLoading = true;
    update();
    final headers = {
      'Authorization': "Bearer ${invoiceHistoryRepo.apiClient.token}",
      'content-type': 'application/pdf',
    };

    String url = "${UrlContainer.baseUrl}${UrlContainer.invoiceDownload}/$invoiceID";
    http.Response response = await http.get(Uri.parse(url), headers: headers);
    final bytesSS = response.body;
    final bytes = response.bodyBytes;
    print(bytesSS);
    print(bytes);
    String extension = "pdf";

    await saveAndOpenFile(bytes, '${MyStrings.appName}_${DateTime.now().millisecondsSinceEpoch}.$extension', extension);

    return;
  }

  Future<void> saveAndOpenFile(List<int> bytes, String fileName, String extension) async {
    Directory? downloadsDirectory;

    if (Platform.isAndroid) {
      await Permission.storage.request();
      downloadsDirectory = Directory('/storage/emulated/0/Download');
    } else if (Platform.isIOS) {
      downloadsDirectory = await getApplicationDocumentsDirectory();
    }

    if (downloadsDirectory != null) {
      final downloadPath = '${downloadsDirectory.path}/$fileName';
      final file = File(downloadPath);
      await file.writeAsBytes(bytes);
      CustomSnackBar.success(successList: ['File saved at: $downloadPath']);
      print('File saved at: $downloadPath');
      await openFile(downloadPath, extension);
    } else {
      CustomSnackBar.error(errorList: ["error"]);
    }
    downloadLoading = false;
    update();
  }

  Future<void> openFile(String path, String extension) async {
    final file = File(path);
    if (await file.exists()) {
      final result = await OpenFile.open(path);
      if (result.type != ResultType.done) {
        if (result.type == ResultType.noAppToOpen) {
          CustomSnackBar.error(errorList: ['File saved at: $path']);
          // CustomSnackBar.error(errorList: [MyStrings.noDocOpenerApp, 'File saved at: $path']);
        }
      }
    } else {
      CustomSnackBar.error(errorList: [MyStrings.fileNotFound]);
    }
  }
}
